# ===================================================================================
# Get-BPLoopbackCheckConfig
# Author: Aleksandar Draskovic
# Date: 2015-05-15
# ===================================================================================

<#
	.SYNOPSIS
		Reads the loopback check configuration


	.DESCRIPTION
		This script retrieves the loopback check configuration. Loopback check is a security feature introduced with Windows Server 2003 SP1.
		
		When you use the fully qualified domain name (FQDN) or a custom host header to browse a local Web site that is hosted on a computer that is running Microsoft Internet Information Services (IIS) 5.1 or a later version, you may receive following error message: "HTTP 401.1 - Unauthorized: Logon Failed". This issue occurs when the Web site uses Integrated Authentication and has a name that is mapped to the local loopback address.


	.EXAMPLE
		Get-BPLoopbackCheckConfig
        
        Retrieves the configuration of the loopback check security feature
    
	.LINK
		https://support.microsoft.com/en-us/kb/896861
#>

param()

function Get-RegistryValue ([string]$path,[string]$valueName)
{
	return (Get-ItemProperty -Path $path -Name $valueName -ErrorAction SilentlyContinue)
}

function Get-LoopbackCheck
{
	Write-Host "Checking method #1 - DisableLoopbackCheck" -ForegroundColor Green
	Write-Host "Reading loopback check configuration... " -ForegroundColor Yellow -NoNewLine
	
	$dlValue = (Get-RegistryValue "HKLM:\System\CurrentControlSet\Control\Lsa" "DisableLoopbackCheck").DisableLoopbackCheck
	if ($dlValue -eq 1)
	{
		Write-Host "Enabled" -ForegroundColor Green
	}
	else
	{
		Write-Host "Disabled" -ForegroundColor Red
	}
	Write-Host ""
}

function Get-LoopbackCheckForHostNames
{
	Write-Host "Checking method #2 - SpecifyHostNames" -ForegroundColor Green
	Write-Host "Reading loopback check configuration (DisableStrictNameChecking)... " -ForegroundColor Yellow -NoNewLine
	
	$dlValue = (Get-RegistryValue "HKLM:\System\CurrentControlSet\Services\LanmanServer\Parameters" "DisableStrictNameChecking").DisableStrictNameChecking
	if ($dlValue -eq 1)
	{
		Write-Host "Enabled" -ForegroundColor Green
	}
	else
	{
		Write-Host "Disabled" -ForegroundColor Red
	}
	
	Write-Host "Reading loopback check configuration (BackConnectionHostNames)... " -ForegroundColor Yellow -NoNewLine
	$registeredURLs = (Get-RegistryValue "HKLM:\System\CurrentControlSet\Control\Lsa\MSV1_0" "BackConnectionHostNames").BackConnectionHostNames
	if ([string]::IsNullOrEmpty($registeredURLs))
	{
		Write-Host "Not present" -ForegroundColor Red
	}
	else
	{
		Write-Host "Enabled" -ForegroundColor Green
	}
	Write-Host ""
}


### Main
Get-LoopbackCheck
Get-LoopbackCheckForHostNames