# ===================================================================================
# Get-BPSSAAuthoritativePages
# Author: Aleksandar Draskovic
# Date: 2015-05-15
# ===================================================================================

<#
	.SYNOPSIS
		Retrieves the number of authoritative pages for all search service applications.


	.DESCRIPTION
		This script retrieves the number of authoritative pages for all search service applications. It reports if the number of authoritative pages is too large (default: 200)
	

	.PARAMETER AuthoritativePagesThreshold
		Defines a threshold in number of authoritative pages. Default: 200 (per level)
		

	.EXAMPLE
		Get-BPSSAAuthoritativePages
        
        Retrieves the health status of all User Profile Service Applications
    
	.LINK
		https://bp.spdockit.com/article/sharepoint-software-boundaries-and-limits/limits-by-feature/search-limits/authoritative-pages/
#>

param(
	[int]$AuthoritativePagesThreshold=200
)

$ssApps = Get-SPEnterpriseSearchServiceApplication
$owner = Get-SPEnterpriseSearchOwner -Level Ssa

Write-Host ""
Write-Host "Checking Search Service Applications... " -NoNewLine

if ($ssApps -ne $null)
{
	Write-Host "$($ssApps.Count) found!" -ForegroundColor Green
	Write-Host "Checking authoritative pages..."
	Write-Host ""
	Write-Host "========================================================================="

	foreach ($ssa in $ssApps)
	{
		Write-Host "Search Service Application: $($ssa.DisplayName)"
		$levels = Get-SPEnterpriseSearchQueryAuthority -SearchApplication $ssa -Owner $owner | group Level | Sort Name
		foreach ($level in $levels)
		{
			if ($level.Count -gt $AuthoritativePagesThreshold)
			{
				Write-Host "Level $($level.Name) contains $($level.Count) pages. Please consider reducing the number of the authoritative pages to achieve better relevance." -ForegroundColor Yellow
			}
			else
			{
				Write-Host "Level $($level.Name) contains $($level.Count) pages." -ForegroundColor Green
			}
		}
		Write-Host "========================================================================="
		Write-Host ""
	}
}
