# ===================================================================================
# Set-BPLoopbackCheckConfig
# Author: Aleksandar Draskovic
# Date: 2015-05-15
# ===================================================================================

<#
	.SYNOPSIS
		Sets the loopback check configuration


	.DESCRIPTION
		This script manages the loopback check configuration. Loopback check is a security feature introduced with Windows Server 2003 SP1.
		
		When you use the fully qualified domain name (FQDN) or a custom host header to browse a local Web site that is hosted on a computer that is running Microsoft Internet Information Services (IIS) 5.1 or a later version, you may receive following error message: "HTTP 401.1 - Unauthorized: Logon Failed". This issue occurs when the Web site uses Integrated Authentication and has a name that is mapped to the local loopback address.
		
		This script configures this security feature behavior.


	.PARAMETER Action
		Action to be performed. Valid values are:
		
		DisableLoopbackCheck
		Completely disables the loopback check feature. Recommended for development and test environments, not recommended for the production environments. 
		
		SpecifyHostNames
		Reads the SharePoint web application URLs and alternate access mapping settings. Selectively disables the loopback check feature for detected URLs. Recommended for the production environments.
		
		Enable
		Enables loopback check

		
	.EXAMPLE
		Set-BPLoopbackCheckConfig -Action DisableLoopbackCheck
        
        Completely disables the loopback check security feature
    
	
    .EXAMPLE
		Set-BPLoopbackCheckConfig -Action SpecifyHostNames
        
        Disables the loopback check configuration for the host names registered with SharePoint. Loopback check feature remains active for all host names and aliases not used by SharePoint.
		

	.EXAMPLE
		Set-BPLoopbackCheckConfig -Action Enable
        
        Enables the loopback check feature (default behavior since Windows Server 2003 SP1)
	
	
	.LINK
		https://support.microsoft.com/en-us/kb/896861
#>


param(
	[Parameter(Mandatory=$true)]
	[ValidateSet("DisableLoopbackCheck","SpecifyHostNames","Enable")]
	[string]$Action
)

function Get-SPWebAppHostNames()
{
	$spWebApps = Get-SPWebApplication
	$urlList = @()
	foreach ($spWebApp in $spWebApps)
	{
		foreach ($url in $spWebApp.AlternateUrls)
		{
			if ($urlList.IndexOf(([System.Uri]$url.IncomingUrl).Host) -eq -1)
			{
				$urlList += ([System.Uri]$url.IncomingUrl).Host
			}
			if ($urlList.IndexOf(([System.Uri]$url.PublicUrl).Host) -eq -1)
			{
				$urlList += ([System.Uri]$url.PublicUrl).Host
			}
		}
	}
	return $urlList
}

function Get-RegistryValue ([string]$path,[string]$valueName)
{
	return (Get-ItemProperty -Path $path -Name $valueName -ErrorAction SilentlyContinue)
}

function Set-RegistryValueDword([string]$path,[string]$valueName,[uint32]$value)
{
	$loopbackCheck = (Get-ItemProperty -Path $path -Name $valueName -ErrorAction SilentlyContinue)
	
	if ($loopbackCheck -eq $null)
	{
		$loopbackCheck = New-ItemProperty -Path $path -Name $valueName -PropertyType DWord -Value $value -ErrorAction SilentlyContinue
	}
	else
	{
		$loopbackCheck = Set-ItemProperty -Path $path -Name $valueName -Value $value -ErrorAction SilentlyContinue -PassThru
	}
}

function Set-RegistryValueMultiString([string]$path,[string]$valueName,$value)
{
	$itemProperty = (Get-ItemProperty -Path $path -Name $valueName -ErrorAction SilentlyContinue)
	
	if ($itemProperty -eq $null)
	{
		$itemProperty = New-ItemProperty -Path $path -Name $valueName -PropertyType MultiString -Value $value -ErrorAction SilentlyContinue
	}
	else
	{
		$itemProperty = Set-ItemProperty -Path $path -Name $valueName -Value $value -ErrorAction SilentlyContinue -PassThru
	}
}

function Restart-IISAdminService()
{
	$title = "Restart IISADMIN service"
	$message = "To apply the changes, we need to restart the IISADMIN service. This can impact the service availability. Do you want to continue?"

	$yes = New-Object System.Management.Automation.Host.ChoiceDescription "&Yes", `
		"Restarts the IISADMIN service."

	$no = New-Object System.Management.Automation.Host.ChoiceDescription "&No", `
		"Exits without restarting the IISADMIN service. You will have to restart the service manually."

	$options = [System.Management.Automation.Host.ChoiceDescription[]]($yes, $no)

	$result = $host.ui.PromptForChoice($title, $message, $options, 0) 

	switch ($result)
	{
		0 { Restart-Service IISADMIN }
		1 { }
	}
}

function Disable-LoopbackCheck([bool]$lcDisable)
{
	$value = [int]$lcDisable
	if ($lcDisable)
	{
		$edString = "Disabling"
	}
	else
	{
		$edString = "Enabling"
	}
	
	Write-Host "$edString loopback check..." -ForegroundColor Green
	
	Set-RegistryValueDword "HKLM:\System\CurrentControlSet\Control\Lsa" "DisableLoopbackCheck" $value
	
	Write-Host "Please restart server to apply changes." -ForegroundColor Red
}

function Disable-LoopbackCheckForHostNames([bool]$lcDisable)
{
	Write-Host "Selectively disabling loopback check by host names is not implemented yet. " -ForegroundColor Yellow 
	$value = [int]$lcDisable
	if ($lcDisable)
	{
		$edString = "Disabling"
	}
	else
	{
		$edString = "Enabling"
	}
	
	Write-Host "$edString loopback check (set host names method)..." -ForegroundColor Green
	
	Set-RegistryValueDword "HKLM:\System\CurrentControlSet\Services\LanmanServer\Parameters" "DisableStrictNameChecking" $value
	if ($lcDisable)
	{
		$registeredURLs = (Get-RegistryValue "HKLM:\System\CurrentControlSet\Control\Lsa\MSV1_0" "BackConnectionHostNames").BackConnectionHostNames
		if ([string]::IsNullOrEmpty($registeredURLs))
		{
			$registeredURLs = @()
		}
		$spUrls = Get-SPWebAppHostNames
		foreach ($spUrl in $spUrls)
		{
			$spUrl = $spUrl.Trim()
			if ($spUrl -ne "")
			{
				if ($registeredURLs.IndexOf($spUrl) -eq -1)
				{
					$registeredURLs += $spUrl
				}
			}
		}
		Set-RegistryValueMultiString "HKLM:\System\CurrentControlSet\Control\Lsa\MSV1_0" "BackConnectionHostNames" $registeredURLs
		Write-Host "If you configured SharePoint for use with Kerberos, you may need to register the following SPNs:" -ForegroundColor Yellow
		foreach ($url in $spUrls)
		{
			Write-Host "HTTP/$url" -ForegroundColor Yellow
		}
		
		# Check if DisableLoopbackCheck is set and remove the setting
		if ((Get-RegistryValue "HKLM:\System\CurrentControlSet\Control\Lsa" "DisableLoopbackCheck").DisableLoopbackCheck -eq 1)
		{
			Write-Host "Detected DisableLoopbackCheck. Deactivating..." -ForegroundColor Green
			Disable-LoopbackCheck $false
		}
	}
	else
	{
		Set-RegistryValueMultiString "HKLM:\System\CurrentControlSet\Control\Lsa\MSV1_0" "BackConnectionHostNames" @()
	}
	
	Restart-IISAdminService
}


### Main
switch ($Action)
{
	"Enable" 					{ Disable-LoopbackCheck $false; Disable-LoopbackCheckForHostNames $false }
	"DisableLoopbackCheck"		{ Disable-LoopbackCheck $true }
	"SpecifyHostNames"			{ Disable-LoopbackCheckForHostNames $true}
}